package com.smfreports.sample;

import java.io.*;
import com.blackhillsoftware.smf.*;
import jakarta.servlet.*;
import jakarta.servlet.http.*;

/**
 * This class does nothing other than receive HTTP post requests and 
 * display the result. It is intended to demonstrate that the
 * data arrives intact at a HTTP server.
 * <p>
 * It caters for 2 types of data:
 * <ol>
 * <li> application/octet-stream : binary data consisting of 1 or more 
 * complete SMF records.
 * <li> application/json : JSON text data e.g. generated by EasySMF-RTI 
 * and EasySMF-JSON
 * </ol>
 * <p>
 * The Jetty Maven plugin can be used to run this servlet:
 * <p>
 * mvn jetty:run -Djetty.http.port=9999
 * <p>
 * This will start the servlet listening on port 9999. 
 * 
 */
public class RtiServlet extends HttpServlet
{
    protected void doPost(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException
    {
        if (!request.getRequestURI().equals("/easysmf"))
        {
            System.out.println("Incorrect path: \"" + request.getRequestURI() + "\", expected \"/easysmf\"");
        }
        else
        {
        	System.out.println("Received post...");
        	
        	switch(request.getContentType())
        	{
        	case "application/octet-stream":
                receiveSmfData(request);   
                response.setStatus(HttpServletResponse.SC_OK);
                break;
        	case "application/json":
                receiveJson(request);
                response.setStatus(HttpServletResponse.SC_OK);   
                break;
                
        	default:
        	    System.out.println("Content type not implemented: " + request.getContentType());
                response.setStatus(HttpServletResponse.SC_OK);   
        	    break;
        	}
        }
    }

    /**
     * Process a POST containing binary SMF data
     * @param request the request
     * @throws IOException if an error is encountered reading the data eg if it is not valid SMF data 
     */
    private void receiveSmfData(HttpServletRequest request) throws IOException 
    {
        // Get the data from the request
        InputStream inputStream = request.getInputStream();
        
        // Read it as a sequence of SMF records 
        try (SmfRecordReader reader = 
                SmfRecordReader.fromStream(inputStream))
        { 
            // print some information from each record
            for (SmfRecord record : reader)
            {
                System.out.format("%24s SMF type: %5d Size: %5d%n", 
                        record.smfDateTime(), 
                        record.recordType(), 
                        record.recordLength());
            }
        }
    }

    /**
     * Process a post containing JSON data
     * @param request the request
     * @throws IOException if an I/O error occurs
     */
    private void receiveJson(HttpServletRequest request) throws IOException 
    {
        // get the request data as a character stream
        try (BufferedReader reader = 
                request.getReader())
        {
            // read and print each line
            String line;
            while((line = reader.readLine()) != null)
            {
                System.out.println(line);
            }
        }
    }
}